package be.kdg.labyrinth.JavaFX.view;

import javafx.geometry.Pos;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.control.Label;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;

public class GameView extends BorderPane {
    public static final Color WALL_COLOR = Color.BLACK;
    public static final Color PATH_COLOR = Color.WHITE;
    public static final Color START_COLOR = Color.GREEN;
    public static final Color SECRET_COLOR = Color.YELLOW;
    public static final Color EXIT_COLOR = Color.RED;
    public static final Color PLAYER_COLOR = Color.BLUE;
    public static final Color TRAIL_COLOR = Color.LIGHTGRAY;

    private Canvas canvas;
    private GraphicsContext gc;
    private Label titleLabel;
    private Label timeLabel;
    private Label scoreLabel;
    private Label playerPositionLabel;
    private int tileSize;
    private int rows, cols;

    public GameView(double width, double height, int rows, int cols, String playerName) {
        this.rows = rows;
        this.cols = cols;
        this.setPrefSize(width, height);
        this.tileSize = calculateTileSize(width * 0.8, height * 0.8, rows, cols);

        // Canvas grootte aanpassen aan het doolhof
        this.canvas = new Canvas(cols * tileSize, rows * tileSize);
        this.gc = canvas.getGraphicsContext2D();

        initializeNodes(playerName);
        layoutNodes();

        // Voeg een listener toe voor het wijzigen van de grootte
        this.widthProperty().addListener((obs, oldVal, newVal) -> resizeCanvas());
        this.heightProperty().addListener((obs, oldVal, newVal) -> resizeCanvas());
    }

    private void resizeCanvas() {
        double newWidth = this.getWidth();
        double newHeight = this.getHeight();
        tileSize = calculateTileSize(newWidth * 0.8, newHeight * 0.8, rows, cols);
        canvas.setWidth(cols * tileSize);
        canvas.setHeight(rows * tileSize);
    }

    private int calculateTileSize(double width, double height, int rows, int cols) {
        int tileWidth = (int) (width / cols);
        int tileHeight = (int) (height / rows);
        return Math.min(tileWidth, tileHeight) - 5;
    }

    private void initializeNodes(String playerName) {
        titleLabel = new Label("Welkom bij het labyrinth " + playerName);
        titleLabel.setStyle("-fx-font-size: 50px; -fx-font-family: 'Berlin Sans FB'; -fx-font-weight: bold");

        timeLabel = new Label("Tijd: 00:00:00");
        timeLabel.setStyle("-fx-font-size: 35px; -fx-text-fill: black; -fx-background-color: white; -fx-border-color: black; -fx-border-width: 2px; -fx-border-radius: 10px");

        scoreLabel = new Label("Scores:\n1. 00:00:00\n2. 00:00:00\n3. 00:00:00\n4. 00:00:00\n5. 00:00:00");
        scoreLabel.setStyle("-fx-font-size: 35px; -fx-text-fill: black; -fx-background-color: white; -fx-border-color: black; -fx-border-width: 2px; -fx-border-radius: 10px");

        playerPositionLabel = new Label("Spelerpositie: (0, 0)");
        playerPositionLabel.setStyle("-fx-font-size: 35px; -fx-text-fill: black; -fx-background-color: white; -fx-border-color: black; -fx-border-width: 2px; -fx-border-radius: 10px");
    }

    private void layoutNodes() {
        VBox topBox = new VBox(25);
        topBox.setAlignment(Pos.CENTER);
        topBox.getChildren().addAll(titleLabel, playerPositionLabel);
        this.setTop(topBox);

        this.setLeft(scoreLabel);

        VBox rightBox = new VBox(10, timeLabel);
        rightBox.setAlignment(Pos.TOP_RIGHT);
        this.setRight(rightBox);

        // StackPane zorgt ervoor dat het labyrinth in het midden staat
        StackPane centerPane = new StackPane(canvas);
        centerPane.setAlignment(Pos.CENTER);
        this.setCenter(centerPane);
    }

    public void drawTile(int row, int col, Color color) {
        gc.setFill(color);
        gc.fillRect(col * tileSize, row * tileSize, tileSize, tileSize);
    }

    public void drawPlayer(int row, int col) {
        gc.setFill(PLAYER_COLOR);
        gc.fillOval(col * tileSize, row * tileSize, tileSize, tileSize);
    }

    public void drawTrail(int row, int col) {
        gc.setFill(TRAIL_COLOR);
        gc.fillRect(col * tileSize, row * tileSize, tileSize, tileSize);
    }

    public void updateTime(String time) {
        timeLabel.setText("Tijd: " + time);
    }

    public void updateScore(String scores) {
        scoreLabel.setText("Scores:\n" + scores);
    }

    public void updatePlayerPosition(int row, int col) {
        playerPositionLabel.setText("Spelerpositie: (" + row + ", " + col + ")");
    }

    public int getTileSize() {
        return tileSize;
    }

    public Canvas getCanvas() {
        return canvas;
    }
}